<?php
/**
 * @package     Joomla.Platform
 * @subpackage  HTTP
 *
 * @copyright   Copyright (C) 2005 - 2012 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE
 */

defined('_JEXEC') or die(); 
use Joomla\CMS\Uri\Uri;

require_once 'httpresponse.php';

/**
 * HTTP client class.
 *
 * @package     Joomla.Platform
 * @subpackage  HTTP
 * @since       11.3
 */
class jscrHttp
{
	/**
	 * @var   Options for the HTTP client.
	 * @since  11.3
	 */
	protected $options;

	/**
	 * @var    jscrHttpTransport  The HTTP transport object to use in sending HTTP requests.
	 * @since  11.3
	 */
	protected $transport;

	/**
	 * Constructor.
	 *
	 * @param   jscrHttpTransport  $transport  The HTTP transport object.
	 *
	 * @since   11.3
	 */
	public function __construct(?jscrHttpTransport $transport = null)
	{ 
		if(is_null($transport) && !class_exists('jscrHttpTransportSocket')) {
			require_once 'transport/httptransportsocket.php';
		}
		$this->transport = isset($transport) ? $transport : new \jscrHttpTransportSocket($this->options);
	}
  
	/**
	 * Method to send the OPTIONS command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request.
	 *
	 * @return  Http\Response
	 *
	 * @since   11.3
	 */
	public function options($url, ?array $headers = null)
	{
		return $this->transport->request('OPTIONS', new Uri($url), null, $headers);
	}

	/**
	 * Method to send the HEAD command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request.
	 *
	 * @return  Http\Response
	 *
	 * @since   11.3
	 */
	public function head($url, ?array $headers = null)
	{
		return $this->transport->request('HEAD', new Uri($url), null, $headers);
	}

	/**
	 * Method to send the GET command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request.
	 *
	 * @return  Http\Response
	 *
	 * @since   11.3
	 */
	public function get($url, ?array $headers = null)
	{
		return $this->transport->request('GET', new Uri($url), null, $headers);
	}

	/**
	 * Method to send the POST command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   mixed   $data     Either an associative array or a string to be sent with the request.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request.
	 *
	 * @return  Http\Response
	 *
	 * @since   11.3
	 */
	public function post($url, $data, ?array $headers = null)
	{
		return $this->transport->request('POST', new Uri($url), $data, $headers);
	}

	/**
	 * Method to send the PUT command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   mixed   $data     Either an associative array or a string to be sent with the request.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request.
	 *
	 * @return  Http\Response
	 *
	 * @since   11.3
	 */
	public function put($url, $data, ?array $headers = null)
	{
		return $this->transport->request('PUT', new Uri($url), $data, $headers);
	}

	/**
	 * Method to send the DELETE command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request.
	 *
	 * @return  Http\Response
	 *
	 * @since   11.3
	 */
	public function delete($url, ?array $headers = null)
	{
		return $this->transport->request('DELETE', new Uri($url), null, $headers);
	}

	/**
	 * Method to send the TRACE command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request.
	 *
	 * @return  Http\Response
	 *
	 * @since   11.3
	 */
	public function trace($url, ?array $headers = null)
	{
		return $this->transport->request('TRACE', new Uri($url), null, $headers);
	}
}
